/***************************************************************************************************
 *
 *  Tanvas, Inc
 *  https://tanvas.co
 *
 *  Copyright(c) 2021. All Rights Reserved
 *
 *  Source code from Tanvas is supplied under the terms of a license agreement
 *  and may not be copied or disclosed except in accordance with the terms of that agreement.
 *  The various license agreements may be found at the Tanvas web site.
 *
 **************************************************************************************************/

#include "haptics.h"
#include "load_png.h"
#include "visuals.h"
#include <SDL.h>
#include <tanvas/tanvastouch.h>

int main(int argc, char** argv)
{
	/* We assume we errored until we know everything's set up properly. */
	int exit_code = 1;

	/* Initialize SDL and show ERROR and CRITICAL log messages by default. */
	SDL_LogSetAllPriority(SDL_LOG_PRIORITY_ERROR);

	if (SDL_Init(SDL_INIT_EVENTS | SDL_INIT_VIDEO) != 0) {
		SDL_LogError(SDL_LOG_CATEGORY_ERROR, "Unable to initialize SDL: %s", SDL_GetError());
		goto error0;
	}

	/* Load the example image and create an SDL surface from it. */
	image background;
	if (!load_rgb_png("catalog_background.png", STBI_rgb, &background)) {
		goto error1;
	}

	SDL_Surface* bg_surface = image_to_sdl_surface(background);
	if (bg_surface == NULL) {
		goto error2;
	}

	/* Create an SDL window that's the same size as the example image. */
	SDL_Window* window = SDL_CreateWindow("Hello Tanvas", SDL_WINDOWPOS_UNDEFINED,
	                                      SDL_WINDOWPOS_UNDEFINED, bg_surface->w, bg_surface->h, 0);
	if (window == NULL) {
		SDL_LogError(SDL_LOG_CATEGORY_ERROR, "Failed to create window: %s", SDL_GetError());
		goto error3;
	}

	/* Connect to the TanvasTouch Engine. */
	tanvastouch_ctx* ctx = connect_to_engine();
	if (ctx == NULL) {
		goto error4;
	}

	/* Create haptic resources. */
	tanvastouch_resource_id view_id = create_haptic_scene(ctx);
	if (view_id == TANVASTOUCH_INVALID_RESOURCE_ID) {
		goto error5;
	}

	/* Move the haptic view to be in the same position as the SDL window. */
	int x, y;
	SDL_GetWindowPosition(window, &x, &y);
	tanvastouch_set_view_position_on_desktop(ctx, view_id, (float)x, (float)y);

	/* Handle UI events. */
	SDL_Surface* window_surface = SDL_GetWindowSurface(window);
	SDL_Event event;
	int running = 1;

	while (running) {
		if (!SDL_WaitEvent(&event)) {
			SDL_LogError(SDL_LOG_CATEGORY_ERROR, "error waiting for events: %s", SDL_GetError());
			break;
		}

		switch (event.type) {
		case SDL_WINDOWEVENT: {
			switch (event.window.event) {
			/* Window needs to be redrawn; draw the example image. */
			case SDL_WINDOWEVENT_EXPOSED:
				SDL_BlitSurface(bg_surface, NULL, window_surface, NULL);
				SDL_UpdateWindowSurface(window);
				break;
			/* Window has moved; move the haptic view to match. */
			case SDL_WINDOWEVENT_MOVED:
				tanvastouch_set_view_position_on_desktop(ctx, view_id, (float)event.window.data1,
				                                         (float)event.window.data2);
				break;
			/* Window has been closed; stop the event loop. */
			case SDL_WINDOWEVENT_CLOSE:
				running = 0;
				break;
			}
		}
		default:
			break;
		}
	}

	exit_code = 0;

	/* Clean up. */
	tanvastouch_destroy_view(ctx, view_id);

error5:
	tanvastouch_close(ctx);

error4:
	SDL_DestroyWindow(window);

error3:
	SDL_FreeSurface(bg_surface);

error2:
	free_rgb_png(&background);

error1:
	SDL_Quit();

error0:
	return exit_code;
}
