/**********************************************************************************************************************
 *
 *  Tanvas, Inc
 *  http://www.tanvas.co
 *
 *  Copyright(c) 2019. All Rights Reserved
 *
 *  Source code from Tanvas is supplied under the terms of a license agreement
 *  and may not be copied or disclosed except in accordance with the terms of that agreement.
 *  The various license agreements may be found at the Tanvas web site.
 *
 *********************************************************************************************************************/

#ifndef TANVAS_TANVASTOUCH_ERRORS_H
#define TANVAS_TANVASTOUCH_ERRORS_H

/**
 * @file tanvastouch_errors.h
 * @brief Status codes used in the TanvasTouch Engine API
 */

/**
 * In a C++ program, uses uniform initialization or old-style initialization to construct
 * tanvastouch_resource_ids and integer error codes.  In a C program, uses C casts.
 * Intended for use only in tanvastouch_errors.h.
 */
#ifdef __cplusplus
#ifdef __cpp_initializer_lists
#define TANVASTOUCH_AS(t, v) (t{v})
#else
#define TANVASTOUCH_AS(t, v) (t(v))
#endif
#else
#define TANVASTOUCH_AS(t, v) ((t)v)
#endif

/**
 * A resource ID that denotes an invalid resource.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_INVALID_RESOURCE_ID TANVASTOUCH_AS(tanvastouch_resource_id, 0)

/**
 * Success status code.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_OK TANVASTOUCH_AS(int, 0)

/* clang-format off */

/**
 * General error category.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR                 (1u << 16u)

/**
 * Errors dealing with haptic resources.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_RESOURCE        (2u << 16u)

/**
 * Errors dealing with the engine's involvement in API calls.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_ENGINE          (3u << 16u)

/**
 * Errors dealing with the application's involvement in API calls.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_CLIENT          (4u << 16u)

/**
 * Errors defined by an API wrapper. These will be defined in the wrapper code. Do not change this
 * value as it'll mean changing all the same in all of the wrappers.
 *
 * @since 3.0.0
 */
#define TANVASTOUCH_ERROR_WRAPPER         (5u << 16u)

/**
 * Errors specific to textures.
 *
 * @ingroup texture
 * @since 4.0.0
 */
#define TANVASTOUCH_ERROR_TEXTURE         (6u << 16u)

/**
 * Errors specific to materials.
 *
 * @ingroup material
 * @since 4.0.2
 */
#define TANVASTOUCH_ERROR_MATERIAL        (7u << 16u)

/**
 * An uncategorized error.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_GENERAL                    TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR | 65535u))

/**
 * A resource ID passed to a API call does not correspond to any resource for the current
 * process.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_RESOURCE_UNKNOWN           TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_RESOURCE | 1u))

/**
 * The requested resource could not be created.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_RESOURCE_CANNOT_CREATE     TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_RESOURCE | 2u))

/**
 * Data was not supplied to an API endpoint that requires it.
 *
 * @since 4.0.0
 */
#define TANVASTOUCH_ERROR_RESOURCE_DATA_REQUIRED     TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_RESOURCE | 3u))

/**
 * @addtogroup texture
 * @{
 */

/**
 * An offset or length used in a set_texture_data_partial call would go out-of-bounds for the given
 * texture.
 *
 * set_texture_data_partial cannot append to or remove from a texture.  To change a texture's size,
 * set new data with set_texture_data.
 *
 * @since 4.0.0
 */
#define TANVASTOUCH_ERROR_TEXTURE_OFFSET_TOO_LARGE   TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_TEXTURE | 1u))

/**
 * A size mismatch was detected when sending texture data (i.e. width * height != length).
 *
 * @since 4.0.0
 */
#define TANVASTOUCH_ERROR_TEXTURE_SIZE_MISMATCH      TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_TEXTURE | 2u))

/**
 * The texture has a negative width.
 *
 * @since 4.0.0
 */
#define TANVASTOUCH_ERROR_TEXTURE_NEGATIVE_WIDTH     TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_TEXTURE | 3u))

/**
 * The texture has a negative height.
 *
 * @since 4.0.0
 */
#define TANVASTOUCH_ERROR_TEXTURE_NEGATIVE_HEIGHT    TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_TEXTURE | 4u))

/** @} */

/**
 * @addtogroup material
 * @{
 */

/**
 * There were too few elements present in the UV array passed to a set_material_uvs call.  (A UV
 * array must contain at least 8 elements.)
 *
 * @since 4.0.2
 */
#define TANVASTOUCH_ERROR_MATERIAL_TOO_FEW_UVS       TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_MATERIAL | 1u))

/**
 * The texture index was invalid.
 *
 * Version 4.0.0 of the TanvasTouch Engine supports up to three textures per material with indices
 * 0, 1, and 2.
 *
 * @since 4.0.2
 */
#define TANVASTOUCH_ERROR_MATERIAL_INVALID_TEXTURE_INDEX \
    TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_MATERIAL | 2u))

/** @} */

/**
 * The API call is not known to the engine, perhaps due to a protocol mismatch between
 * the engine and application.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_ENGINE_UNKNOWN_CALL        TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_ENGINE | 1u))

/**
 * The data comprising the API call is in the correct format but failed engine-side verification,
 * perhaps due to a corrupt buffer or transmission problem.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_ENGINE_FAILED_VERIFICATION TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_ENGINE | 2u))

/**
 * The data comprising the API call is not in the correct format.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_ENGINE_INVALID_REQUEST     TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_ENGINE | 3u))

/**
 * The API call failed engine-side due to an uncategorized error.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_ENGINE_OPERATION_FAILED    TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_ENGINE | 4u))

/**
 * The API call failed engine-side because the requested operation is already running.
 *
 * @since 3.0.0
 */
#define TANVASTOUCH_ERROR_ENGINE_ALREADY_RUNNING     TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_ENGINE | 5u))

/**
 * The engine's response to an API call is not in the correct format.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_CLIENT_INVALID_RESPONSE    TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_CLIENT | 1u))

/**
 * The engine's response to an API call does not have the expected sequence number.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_CLIENT_INVALID_SEQUENCE    TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_CLIENT | 2u))

/**
 * The engine's response to an API call failed the API library's verification.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_CLIENT_FAILED_VERIFICATION TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_CLIENT | 3u))

/**
 * The API library cannot connect to the engine, perhaps because the engine is not running
 * or is listening on a different endpoint.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_CLIENT_CANNOT_FIND_ENGINE  TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_CLIENT | 4u))

/**
 * The API library was denied permission to connect to the given endpoint.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_CLIENT_PERMISSION_DENIED   TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_CLIENT | 5u))

/**
 * The version of the API in use by the library is not compatible with the API supported by the
 * engine.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_CLIENT_INCOMPATIBLE        TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_CLIENT | 6u))

/**
 * The client is not licensed - either because the verification is still in progress, or the license
 * is denied.
 *
 * @since 3.0.0
 * @deprecated Version 4.0.0 of the TanvasTouch Engine removed client licensing; therefore, this
 * 	error is no longer issued.
 */
#define TANVASTOUCH_ERROR_CLIENT_UNLICENSED          TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_CLIENT | 7u))

/**
 * The client has not been correctly configured.
 *
 * @since 3.0.0
 */
#define TANVASTOUCH_ERROR_CLIENT_NOT_INITIALIZED     TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_CLIENT | 8u))

/**
 * The API library could not connect to the engine due to an uncategorized error.
 *
 * @since 2.0.0
 */
#define TANVASTOUCH_ERROR_CLIENT_GENERAL             TANVASTOUCH_AS(int, (TANVASTOUCH_ERROR_CLIENT | 65535u))
/* clang-format on */

#endif /* TANVAS_TANVASTOUCH_ERRORS_H */
