/***************************************************************************************************
 *
 *  Tanvas, Inc
 *  https://tanvas.co
 *
 *  Copyright(c) 2021. All Rights Reserved
 *
 *  Source code from Tanvas is supplied under the terms of a license agreement
 *  and may not be copied or disclosed except in accordance with the terms of that agreement.
 *  The various license agreements may be found at the Tanvas web site.
 *
 **************************************************************************************************/

#include "haptics.h"
#include "load_png.h"
#include <SDL.h>

#define CHECK(expr, errlabel)                                                                      \
	{                                                                                              \
		int rc = (expr);                                                                           \
		if (rc != TANVASTOUCH_OK) {                                                                \
			SDL_LogError(SDL_LOG_CATEGORY_ERROR, #expr " failed: %s", tanvastouch_strerror(rc));   \
			goto errlabel;                                                                         \
		}                                                                                          \
	}

tanvastouch_ctx* connect_to_engine()
{
	tanvastouch_ctx* ctx;
	CHECK(tanvastouch_open(NULL, &ctx), error);

	return ctx;

error:
	return NULL;
}

tanvastouch_resource_id create_haptic_scene(tanvastouch_ctx* ctx)
{
	tanvastouch_resource_id view_id = TANVASTOUCH_INVALID_RESOURCE_ID;

	/* Load data for the haptic texture. */
	image surface;
	if (!load_rgb_png("catalog_haptics.png", STBI_grey, &surface)) {
		goto error;
	}

	/* Create, position, and size the sprite. */
	tanvastouch_resource_id sprite_id = TANVASTOUCH_INVALID_RESOURCE_ID;
	CHECK(tanvastouch_create_sprite(ctx, &sprite_id), error);
	CHECK(tanvastouch_set_sprite_position(ctx, sprite_id, 0, 0), error);
	CHECK(tanvastouch_set_sprite_size(ctx, sprite_id, surface.width, surface.height), error);

	/* Create the texture and link it to the sprite. */
	tanvastouch_resource_id material_id = TANVASTOUCH_INVALID_RESOURCE_ID;
	CHECK(tanvastouch_create_material(ctx, &material_id), error);
	CHECK(tanvastouch_set_sprite_material(ctx, sprite_id, material_id), error);

	tanvastouch_resource_id texture_id = TANVASTOUCH_INVALID_RESOURCE_ID;
	CHECK(tanvastouch_create_texture(ctx, &texture_id), error);
	CHECK(tanvastouch_add_material_texture(ctx, material_id, 0, texture_id), error);
	CHECK(tanvastouch_set_material_wrapping_mode(ctx, material_id, 0, 0), error);
	CHECK(
	    tanvastouch_set_texture_data(ctx, texture_id, surface.data, surface.width, surface.height),
	    error);

	/* Create a haptic view and add the sprite to it.
	 *
	 * In this example, this step is done last because the view ID is used to signal success
	 * or failure from this function.   Another way to do this might be to use an error code
	 * as the return value and use an out parameter to return the view ID.
	 */
	CHECK(tanvastouch_create_view(ctx, &view_id), error);
	CHECK(tanvastouch_set_view_size(ctx, view_id, surface.width, surface.height), error);
	CHECK(tanvastouch_set_view_enabled(ctx, view_id, true), error);
	CHECK(tanvastouch_add_sprite(ctx, view_id, sprite_id), error);

error:
	free_rgb_png(&surface);
	return view_id;
}
