/***************************************************************************************************
 *
 *  Tanvas, Inc
 *  https://tanvas.co
 *
 *  Copyright(c) 2019. All Rights Reserved
 *
 *  Source code from Tanvas is supplied under the terms of a license agreement
 *  and may not be copied or disclosed except in accordance with the terms of that agreement.
 *  The various license agreements may be found at the Tanvas web site.
 *
 **************************************************************************************************/

#ifndef TANVAS_TANVASTOUCH_H
#define TANVAS_TANVASTOUCH_H

/**
 * @file tanvastouch.h
 * @brief The TanvasTouch Engine C API
 *
 * This file contains all functions in the TanvasTouch C API.
 * For error codes, see tanvastouch_errors.h.
 *
 * @mainpage
 * @section Overview
 *
 * The TanvasTouch Engine C API manages haptic resources, which are used by the TanvasTouch
 * Engine to direct the TanvasTouch Controller and touchscreen to render haptic textures.
 * The TanvasTouch hardware renders textures by modulating friction under the operator's fingers.
 *
 * @section usage Getting started
 *
 * All C API functions return a status code; status codes are documented in tanvastouch_errors.h.
 * Most C API functions have one or more out-parameters that return resource IDs or attributes.
 *
 * To use the C API, first open a context:
 *
 * @code
 *      tanvastouch_ctx *ctx = NULL;
 *      int rc = tanvastouch_open(engine_path, &ctx);
 *
 *      if (rc != TANVASTOUCH_OK) {
 *          fprintf(stderr, "error: %s\n", tanvastouch_strerror(rc));
 *          tanvastouch_close(ctx);
 *          return -1;
 *      }
 * @endcode
 *
 * @warning To check if a function succeeded, you **must compare its result to #TANVASTOUCH_OK.**
 * Checking for a nonzero result (`if (rc) { ... }`) is **not** correct.
 *
 * With this context, you can create haptic resources.  Each process has its own collection of
 * haptic resources; these cannot be shared across processes.
 *
 * @code
 *      tanvastouch_resource_id view_id = TANVASTOUCH_INVALID_RESOURCE_ID;
 *      int rc = tanvastouch_create_view(ctx, &view_id);
 *
 *      if (rc != TANVASTOUCH_OK) {
 *          fprintf(stderr, "error: %s\n", tanvastouch_strerror(rc));
 *          tanvastouch_close(ctx);
 *          return -1;
 *      }
 * @endcode
 *
 * Finally, close the context when it's no longer needed:
 *
 * @code
 *      tanvastouch_close(ctx);
 *      return 0;
 * @endcode
 *
 * @section coordinate-system Haptic resources are positioned using the window system's rules
 *
 * The coordinate system for haptic resources is provided by the window system.  Details for
 * supported systems are given below.  The Engine works in either logical or physical pixels
 * depending on what is most convenient for mapping touch inputs to positions on the virtual
 * desktop.
 *
 * @subsection coordinate-system-windows-10 The coordinate system for Windows 10
 *
 * - X increases to the right and Y increases going down.  For views, the point (0, 0) is at
 *   top-left corner of the main display.  For sprites, the point (0, 0) is the top-left corner
 *   of their parent view.
 * - The Engine runs in Per-Monitor v2 DPI awareness mode and therefore operates in physical
 *   pixels.
 *
 * It is recommended that applications interacting with the Engine also use Per-Monitor V2 or
 * Per-Monitor DPI awareness modes.  For more information, see the MSDN article
 * "High DPI Desktop Application Development on Windows":
 * https://docs.microsoft.com/en-us/windows/win32/hidpi/high-dpi-desktop-application-development-on-windows
 *
 * @section notes Notes on API behavior
 * @subsection resource-status Resource-related API call return value
 *
 * API calls that set or fetch properties of haptic resources typically return the following values:
 *
 * - #TANVASTOUCH_OK: Set or fetch succeeded
 * - #TANVASTOUCH_ERROR_RESOURCE_UNKNOWN: A resource ID passed to the API function does not map
 *     to a known resource for the calling process
 * - #TANVASTOUCH_ERROR_ENGINE_OPERATION_FAILED: The engine could not apply the requested change
 *
 * @subsection out-param-behavior Out-parameters are only written on success
 *
 * Most of the API calls that write data to out-parameters will write data if and only if the
 * return value of the call is #TANVASTOUCH_OK.  On failure, the original contents of the pointed-to
 * storage will not be modified.
 *
 * @subsection resource-ids-are-not-cleared Destructors do not clear resource IDs
 *
 * Destructor functions do not clear application-side copies of resource IDs.  It is safe to keep
 * these IDs around, but any modifications or queries done with those IDs will return
 * #TANVASTOUCH_ERROR_RESOURCE_UNKNOWN.
 */

#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <tanvas/tanvastouch_errors.h>

#ifdef __cplusplus
extern "C" {
#endif

#if defined(TANVASTOUCH_STATIC)
#define TANVASTOUCH_API
#else
#if defined(TANVASTOUCH_EXPORTS_DLL)
#ifdef _WIN32
#define TANVASTOUCH_API __declspec(dllexport)
#elif defined(__GNUC__)
#define TANVASTOUCH_API __attribute__((visibility("default")))
#else
#error "Don't know the convention for exporting symbols from a DLL/shared object"
#endif
#else
#ifdef _WIN32
#define TANVASTOUCH_API __declspec(dllimport)
#elif defined(__GNUC__)
#define TANVASTOUCH_API
#else
#error "Don't know the convention for importing symbols from a DLL/shared object"
#endif
#endif
#endif

/**
 * @addtogroup datatype Data types
 * @{
 */

/**
 * The API context.
 *
 * A context is constructed with tanvastouch_open() and destructed with tanvastouch_close().
 * Each thread needs its own context.
 */
typedef struct tanvastouch_ctx tanvastouch_ctx;

/**
 * A resource ID.
 *
 * This should be treated as an opaque handle.
 *
 * @see #TANVASTOUCH_INVALID_RESOURCE_ID
 */
typedef uint64_t tanvastouch_resource_id;

/**
 * Return a string corresponding to a TANVASTOUCH_ERROR_* error code.
 *
 * The returned string is owned by the TanvasTouch API library; do not free it.
 *
 * This function never returns NULL.
 *
 * @param err The code to look up.
 * @return An error string for the given code.
 */
TANVASTOUCH_API const char* tanvastouch_strerror(int err);

/**
 * Check whether the given resource ID is not #TANVASTOUCH_INVALID_RESOURCE_ID.
 *
 * @warning This function does not communicate with the engine; therefore, it cannot determine
 * whether a resource ID really corresponds to a resource.  This may be changed in future
 * versions of the API.
 *
 * @param id The ID to check.
 * @return True if the ID is not #TANVASTOUCH_INVALID_RESOURCE_ID, false otherwise.
 */
TANVASTOUCH_API bool tanvastouch_is_valid(tanvastouch_resource_id id);

/** @} */

/**
 * @addtogroup context Context creation and destruction
 * @{
 */

/**
 * Opens a connection to the TanvasTouch Engine.
 *
 * This function may be used to either acquire a connection or test that a connection can be made
 * to an engine.  To acquire a connection, pass a non-NULL pointer in p_ctx; to test, pass NULL
 * in p_ctx.
 *
 * You should always call #tanvastouch_close on the pointer passed in p_ctx.
 *
 * Failure codes:
 *
 * - #TANVASTOUCH_ERROR_CLIENT_CANNOT_FIND_ENGINE
 * - #TANVASTOUCH_ERROR_CLIENT_PERMISSION_DENIED
 * - #TANVASTOUCH_ERROR_CLIENT_INCOMPATIBLE
 *
 * Examples:
 *
 * Testing for an engine at the default location:
 *
 * @code
 *     int rc = tanvastouch_open(NULL, NULL, NULL, NULL);
 *
 *     if (rc == TANVASTOUCH_OK) {
 *         handle_connection_ok();
 *     } else {
 *         handle_connection_error(rc);
 *     }
 * @endcode
 *
 * Establishing a connection:
 *
 * @code
 *      tanvastouch_ctx *ctx = NULL;
 *      int rc = tanvastouch_open(engine_path, &ctx);
 *
 *      if (rc != TANVASTOUCH_OK) {
 *          fprintf(stderr, "error: %s\n", tanvastouch_strerror(rc));
 *          tanvastouch_close(ctx);
 *          return -1;
 *      }
 * @endcode
 *
 * @param engine_name The engine to connect to.  May be NULL.
 * 	    On UNIX platforms, this is the path to a UNIX socket.
 * 	    On Windows, this is currently unused.
 * 	    If this is NULL, the default name will be used.
 * @param p_ctx If not null, a pointer will be written to this location.
 *      On success, the pointed-to location will contain a pointer to a tanvastouch_ctx object.
 *      On failure, the pointed-to location will contain NULL.
 * @return #TANVASTOUCH_OK on success, a failure code otherwise.
 */
TANVASTOUCH_API int tanvastouch_open(const char* engine_name, tanvastouch_ctx** p_ctx);

/**
 * Destroys contexts created by #tanvastouch_open.
 *
 * @param ctx The context to destroy.  The contents of the pointed-to memory should not be read
 *      after destruction.  If ctx is NULL, this function does nothing.
 * @return #TANVASTOUCH_OK.  At present, this function does not return any failure codes.
 */
TANVASTOUCH_API int tanvastouch_close(tanvastouch_ctx* ctx);

/**
 * The logging callback function type.
 *
 * @param data User data pointer.
 * @param str The log string.  This string lives only as long as the callback
 *      invocation; if you need it to live longer, you must copy it.
 */
typedef void (*tanvastouch_log_cb)(void* data, const char* str);

/**
 * Set a logger callback.
 *
 * Some API operations may produce extra log output as a debugging aid; this callback
 * receives those log messages.  The log output is probably not of interest to
 * application users, but it will not contain any sensitive information,
 * so the visibility of the logging output is at your discretion.
 *
 * Each context has its own logger callback.  The default callback is the null pointer,
 * i.e. the callback is disabled.
 *
 * The context does not take ownership of the user data pointer.
 *
 * @param ctx The context to modify.
 * @param cb The callback to assign.
 * @param data User data pointer.
 * @return A pointer to the old callback.
 */
TANVASTOUCH_API tanvastouch_log_cb tanvastouch_set_log_cb(tanvastouch_ctx* ctx,
                                                          tanvastouch_log_cb cb, void* data);

/** @} */

/**
 * @addtogroup view Views
 * @{
 */

/**
 * Create a view.
 *
 * A view starts in an enabled state.  The view is initially positioned at (0, 0), and has a size
 * of (INT_MAX, INT_MAX).
 *
 * @see \ref coordinate-system
 * @see \ref out-param-behavior
 * @param ctx The context to use.
 * @param p_view_id Storage for the new view's ID.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_create_view(tanvastouch_ctx* ctx,
                                            tanvastouch_resource_id* p_view_id);

/**
 * Destroy a view.
 *
 * Destroying views does not destroy any sprites contained in that view.
 *
 * @see \ref resource-ids-are-not-cleared
 * @param ctx The context to use.
 * @param view_id The view to destroy.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_destroy_view(tanvastouch_ctx* ctx, tanvastouch_resource_id view_id);

/**
 * Set whether the given view will participate in haptic rendering.
 *
 * @param ctx The context to use.
 * @param view_id The view to modify.
 * @param enabled True to enable, false to disable.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_set_view_enabled(tanvastouch_ctx* ctx,
                                                 tanvastouch_resource_id view_id, bool enabled);

/**
 * Get whether a view will participate in haptic rendering.
 *
 * @see \ref out-param-behavior
 * @param ctx The context to use.
 * @param view_id The view to query.
 * @param p_enabled Storage for the status.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_is_view_enabled(tanvastouch_ctx* ctx,
                                                tanvastouch_resource_id view_id, bool* p_enabled);

/**
 * Set a view's position.
 *
 * @warning Future releases of the engine may change views so that they use the coordinate system
 * of the application window; i.e. (0, 0) may become the top-left corner of the application window.
 * If this happens, a new function to set view position will be introduced, and this function will
 * become deprecated.
 *
 * @see \ref coordinate-system
 * @param ctx The context to use.
 * @param view_id The view to modify.
 * @param x X position.
 * @param y Y position.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_set_view_position_on_desktop(tanvastouch_ctx* ctx,
                                                             tanvastouch_resource_id view_id,
                                                             float x, float y);

/**
 * Get a view's position.
 *
 * @see \ref coordinate-system
 * @param ctx The context to use.
 * @param view_id The view to query.
 * @param p_x Storage for the X position.
 * @param p_y Storage for the Y position.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_get_view_position(tanvastouch_ctx* ctx,
                                                  tanvastouch_resource_id view_id, float* p_x,
                                                  float* p_y);

/**
 * Set a view's size.
 *
 * @warning As views are positioned relative to the desktop, views may extend outside the bounds
 * of the application window.  Currently, the engine will render haptics in these out-of-bounds
 * regions, but this behavior will be eliminated in future versions.
 *
 * @see \ref coordinate-system
 * @param ctx The context to use.
 * @param view_id The view to modify.
 * @param width Desired width.
 * @param height Desired height.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_set_view_size(tanvastouch_ctx* ctx, tanvastouch_resource_id view_id,
                                              int width, int height);

/**
 * Get a view's size.
 *
 * @see \ref coordinate-system
 * @see \ref out-param-behavior
 * @param ctx The context to use.
 * @param view_id The view to query.
 * @param p_width Width storage.
 * @param p_height Height storage.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_get_view_size(tanvastouch_ctx* ctx, tanvastouch_resource_id view_id,
                                              int* p_width, int* p_height);

/** @} */

/**
 * @addtogroup sprite Sprites
 * @{
 */

/**
 * Add a sprite to a view.
 *
 * The sprite is positioned in its view's coordinate system; (0, 0) is the top-left corner of its
 * parent view.
 *
 * The view does not take ownership of the sprite: if the view is destroyed, the sprite will
 * continue to exist.
 *
 * @see \ref coordinate-system
 * @param ctx The context to use.
 * @param view_id The view to modify.
 * @param sprite_id The sprite to add to the view.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_add_sprite(tanvastouch_ctx* ctx, tanvastouch_resource_id view_id,
                                           tanvastouch_resource_id sprite_id);

/**
 * Remove a sprite from a view.
 *
 * @param ctx The context to use.
 * @param view_id The view to modify.
 * @param sprite_id The sprite to remove from the view.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_remove_sprite(tanvastouch_ctx* ctx, tanvastouch_resource_id view_id,
                                              tanvastouch_resource_id sprite_id);

/**
 * Get whether a sprite is contained in a view.
 *
 * @see \ref out-param-behavior
 * @param ctx The context to use.
 * @param view_id The view to query.
 * @param sprite_id The sprite to query.
 * @param p_contains Is-contained storage.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_contains_sprite(tanvastouch_ctx* ctx,
                                                tanvastouch_resource_id view_id,
                                                tanvastouch_resource_id sprite_id,
                                                bool* p_contains);

/**
 * Get the number of sprites in a view.
 *
 * @param ctx The context to use.
 * @param view_id The view to query.
 * @param p_count Storage for the result.
 * @return \ref resource-status
 * @since 4.1.0
 */
TANVASTOUCH_API int tanvastouch_get_sprite_count(tanvastouch_ctx* ctx,
                                                 tanvastouch_resource_id view_id, int* p_count);

/**
 * Create a sprite.
 *
 * A sprite starts at (0, 0) with size (0, 0) and a rotation of zero radians, with the pivot point
 * at (0, 0).  A sprite does not have an initial view.
 *
 * The sprite is positioned in its view's coordinate system; (0, 0) is the top-left corner of its
 * parent view.
 *
 * @see \ref coordinate-system
 * @see \ref out-param-behavior
 * @param ctx The context to use.
 * @param p_sprite_id Sprite ID storage.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_create_sprite(tanvastouch_ctx* ctx,
                                              tanvastouch_resource_id* p_sprite_id);

/**
 * Destroy a sprite.
 *
 * The sprite ID is not modified by this function.  It is safe to keep the sprite ID around, but any
 * modifications or queries done with that ID will return TANVASTOUCH_ERROR_RESOURCE_UNKNOWN.
 *
 * Destroying sprites does not destroy any materials or textures contained in that sprite.
 *
 * @param ctx The context to use.
 * @param sprite_id The sprite to destroy.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_destroy_sprite(tanvastouch_ctx* ctx,
                                               tanvastouch_resource_id sprite_id);

/**
 * Set whether the given sprite will participate in haptic rendering.
 *
 * @see \ref resource-ids-are-not-cleared
 * @param ctx The context to use.
 * @param sprite_id The sprite to modify.
 * @param enabled True to enable, false to disable.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_set_sprite_enabled(tanvastouch_ctx* ctx,
                                                   tanvastouch_resource_id sprite_id, bool enabled);

/**
 * Get whether a sprite is enabled.
 *
 * @see \ref out-param-behavior
 * @param ctx The context to use.
 * @param sprite_id The sprite to query.
 * @param p_enabled Is-enabled storage.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_is_sprite_enabled(tanvastouch_ctx* ctx,
                                                  tanvastouch_resource_id sprite_id,
                                                  bool* p_enabled);

/**
 * Set a sprite's position.
 *
 * The origin of the sprite's coordinate system is the top-left corner of its parent view.
 * The origin of the sprite is its top-left corner.
 *
 * @see \ref coordinate-system
 * @param ctx The context to use.
 * @param sprite_id The sprite to modify.
 * @param x X position.
 * @param y Y position.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_set_sprite_position(tanvastouch_ctx* ctx,
                                                    tanvastouch_resource_id sprite_id, float x,
                                                    float y);

/**
 * Get the sprite's position.
 *
 * @see \ref coordinate-system
 * @see \ref out-param-behavior
 * @param ctx The context to use.
 * @param sprite_id The sprite to query.
 * @param p_x X-position storage.
 * @param p_y Y-position storage.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_get_sprite_position(tanvastouch_ctx* ctx,
                                                    tanvastouch_resource_id sprite_id, float* p_x,
                                                    float* p_y);

/**
 * Set a sprite's size.
 *
 * @see \ref coordinate-system
 * @param ctx The context to use.
 * @param sprite_id The sprite to modify.
 * @param width Desired width.
 * @param height Desired height.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_set_sprite_size(tanvastouch_ctx* ctx,
                                                tanvastouch_resource_id sprite_id, int width,
                                                int height);

/**
 * Get a sprite's size.
 *
 * @see \ref coordinate-system
 * @see \ref out-param-behavior
 * @param ctx The context to use.
 * @param sprite_id The sprite to query.
 * @param p_width Width storage.
 * @param p_height Height storage.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_get_sprite_size(tanvastouch_ctx* ctx,
                                                tanvastouch_resource_id sprite_id, int* p_width,
                                                int* p_height);

/**
 * Set a sprite's rotation and pivot point.
 *
 * The rotation angle is specified in radians; rotation proceeds clockwise.  The pivot point
 * is specified in the coordinate system of the sprite's view.
 *
 * @see \ref coordinate-system
 * @param ctx The context to use.
 * @param sprite_id The sprite to modify.
 * @param theta Sprite angle.
 * @param pivot_x Pivot point X.
 * @param pivot_y Pivot point Y.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_set_sprite_rotation(tanvastouch_ctx* ctx,
                                                    tanvastouch_resource_id sprite_id, float theta,
                                                    float pivot_x, float pivot_y);

/**
 * Return a sprite's angle.
 *
 * The rotation angle is specified in radians; rotation proceeds clockwise.
 *
 * @param ctx The context to use.
 * @param sprite_id The sprite to modify.
 * @param p_theta Angle storage.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_get_sprite_theta(tanvastouch_ctx* ctx,
                                                 tanvastouch_resource_id sprite_id, float* p_theta);

/**
 * Return a sprite's pivot point.
 *
 * The pivot point is specified in the coordinate system of the sprite's view.
 *
 * @see \ref coordinate-system
 * @param ctx The context to use.
 * @param sprite_id The sprite to query.
 * @param p_pivot_x X-position storage.
 * @param p_pivot_y Y-position storage.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_get_sprite_pivot(tanvastouch_ctx* ctx,
                                                 tanvastouch_resource_id sprite_id,
                                                 float* p_pivot_x, float* p_pivot_y);

/**
 * Set a sprite's material.
 *
 * @param ctx The context to use.
 * @param sprite_id The sprite to modify.
 * @param material_id The material to assign.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_set_sprite_material(tanvastouch_ctx* ctx,
                                                    tanvastouch_resource_id sprite_id,
                                                    tanvastouch_resource_id material_id);

/**
 * Get the material ID set for the sprite.
 *
 * @param ctx The context to use.
 * @param sprite_id The sprite to query.
 * @param p_material_id On success, will contain the material ID set for the sprite, or
 *     TANVASTOUCH_INVALID_RESOURCE_ID if the sprite does not have a material set.
 * @since 4.1.0
 */
TANVASTOUCH_API int tanvastouch_get_sprite_material(tanvastouch_ctx* ctx,
                                                    tanvastouch_resource_id sprite_id,
                                                    tanvastouch_resource_id* p_material_id);

/**
 * Clears a sprite's material.
 *
 * If successful, subsequent calls to tanvastouch_get_sprite_material on the affected sprite
 * will return TANVASTOUCH_INVALID_RESOURCE_ID.
 *
 * @param ctx The context to use.
 * @param sprite_id The sprite to modify.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_remove_sprite_material(tanvastouch_ctx* ctx,
                                                       tanvastouch_resource_id sprite_id);

/**
 * Set a sprite's depth.
 *
 * Depth zero is the closest to the screen; increasing depth means going deeper into the screen.
 *
 * @param ctx The context to use.
 * @param sprite_id The sprite to modify.
 * @param depth Desired depth.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_set_sprite_depth(tanvastouch_ctx* ctx,
                                                 tanvastouch_resource_id sprite_id, float depth);

/**
 * Get a sprite's depth.
 *
 * @see \ref out-param-behavior
 * @param ctx The context to use.
 * @param sprite_id The sprite to query.
 * @param p_depth Depth storage.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_get_sprite_depth(tanvastouch_ctx* ctx,
                                                 tanvastouch_resource_id sprite_id, float* p_depth);

/** @} */

/**
 * @addtogroup material Materials
 * @{
 */

/**
 * Create a material.
 *
 * A material has neither an initial sprite nor texture.
 *
 * The default UV coordinates for a material maps all textures onto the material's sprite, i.e.
 * they are [(0, 0), (1, 0), (1, 1), (0, 1)].
 *
 * The default wrapping mode is tanvastouch_wrap_mode_clamp.
 *
 * @see \ref out-param-behavior
 * @see \ref uv-coords
 * @param ctx The context to use.
 * @param p_material_id Material ID storage.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_create_material(tanvastouch_ctx* ctx,
                                                tanvastouch_resource_id* p_material_id);

/**
 * Destroy a material.
 *
 * Destroying a material does not destroy any attached textures.
 *
 * @see \ref resource-ids-are-not-cleared
 * @param ctx The context to use.
 * @param material_id The material to destroy.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_destroy_material(tanvastouch_ctx* ctx,
                                                 tanvastouch_resource_id material_id);

/**
 * Add a texture to a material.
 *
 * The default UV coordinates for a material's texture maps the full texture onto the material's
 * sprite, i.e. they are [(0, 0), (1, 0), (1, 1), (0, 1)].
 *
 * A material may currently have a maximum of 3 textures.
 *
 * @see \ref uv-coords
 * @param ctx The context to use.
 * @param material_id The material to modify.
 * @param index The index to use; starts at zero.
 * @param texture_id The texture to attach.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_add_material_texture(tanvastouch_ctx* ctx,
                                                     tanvastouch_resource_id material_id, int index,
                                                     tanvastouch_resource_id texture_id);

/**
 * Remove a texture from a material.
 *
 * If a material has (say) two textures at indices 0 and 1, it is safe to remove the texture at
 * index 0 without disturbing the texture at index 1.  Removing textures does not change the indices
 * of other textures.
 *
 * @param ctx The context to use.
 * @param material_id The material to modify.
 * @param index The index to remove.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_remove_material_texture(tanvastouch_ctx* ctx,
                                                        tanvastouch_resource_id material_id,
                                                        int index);

/**
 * Retrieve the texture at the given index in a material.
 *
 * If there is no texture at the given index, returns TANVASTOUCH_INVALID_RESOURCE_ID.
 *
 * @param ctx The context to use.
 * @param material_id The material to query.
 * @param index The index in the material to query.
 * @param p_texture_id Storage for the texture ID.
 * @return \ref resource-status
 * @since 4.1.0
 */
TANVASTOUCH_API int tanvastouch_get_material_texture(tanvastouch_ctx* ctx,
                                                     tanvastouch_resource_id material_id, int index,
                                                     tanvastouch_resource_id* p_texture_id);

/**
 * Set the UV coordinates of a texture in a material.
 *
 * @section uv-coords UV coordinate storage
 *
 * The UV coordinates define how the texture is mapped to the material's sprite.  The coordinates
 are
 * supplied as an array of eight floating-point numbers which are interpreted as follows:
 *
 * @verbatim
   [x1, y1,   x2, y2,    x3, y3,       x4, y4       ]
    top-left  top-right  bottom-right  bottom-left
   @endverbatim
 *
 * The UV coordinates array must contain at least eight elements; if fewer elements are supplied, an
 * out-of-bounds read will occur.
 *
 * The default UV coordinates map the entire texture onto the sprite.
 *
 * @param ctx The context to use.
 * @param material_id The material to modify.
 * @param index The texture index to modify.
 * @param p_uv The new UV coordinates.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_set_material_uv(tanvastouch_ctx* ctx,
                                                tanvastouch_resource_id material_id, int index,
                                                float* p_uv);

/**
 * Get the UV coordinates for a material's texture.
 *
 * p_uv must point to storage with enough space for at least eight floats.
 *
 * @see \ref out-param-behavior
 * @see \ref uv-coords
 * @param ctx The context to use.
 * @param material_id The material to query.
 * @param index The texture index to query.
 * @param p_uv UV coordinate storage.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_get_material_uv(tanvastouch_ctx* ctx,
                                                tanvastouch_resource_id material_id, int index,
                                                float* p_uv);

/**
 * The wrapping mode for a material's texture.
 */
enum tanvastouch_wrap_mode_e {
	/**
	 * Clamps UV coordinates between 0 and 1, extending a texture's edges to the sprite's edges.
	 */
	tanvastouch_wrap_mode_clamp = 0,

	/**
	 * Tile the texture across the sprite.
	 */
	tanvastouch_wrap_mode_tile = 1,
};

typedef enum tanvastouch_wrap_mode_e tanvastouch_wrap_mode;

/**
 * Set the wrapping mode for a material's texture.
 *
 * @param ctx The context to use.
 * @param material_id The material to modify.
 * @param index The texture index to modify.
 * @param mode The wrapping mode; see #tanvastouch_wrap_mode_e for acceptable values.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_set_material_wrapping_mode(tanvastouch_ctx* ctx,
                                                           tanvastouch_resource_id material_id,
                                                           int index, int mode);

/**
 * Get the wrapping mode for a material's texture.
 *
 * @param ctx The context to use.
 * @param material_id The material to query.
 * @param index The texture index to query.
 * @param p_mode Mode storage; see #tanvastouch_wrap_mode_e for acceptable values.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_get_material_wrapping_mode(tanvastouch_ctx* ctx,
                                                           tanvastouch_resource_id material_id,
                                                           int index, int* p_mode);

/** @} */

/**
 * @addtogroup texture Textures
 * @{
 */

/**
 * Create a texture.
 *
 * A texture has no initial data.
 *
 * @see \ref out-param-behavior
 * @param ctx The context to use.
 * @param p_texture_id Texture ID storage.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_create_texture(tanvastouch_ctx* ctx,
                                               tanvastouch_resource_id* p_texture_id);

/**
 * Destroy a texture.
 *
 * @see \ref resource-ids-are-not-cleared
 * @param ctx The context to use.
 * @param texture_id The texture to destroy.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_destroy_texture(tanvastouch_ctx* ctx,
                                                tanvastouch_resource_id texture_id);

/**
 * Set a texture's data.
 *
 * Texture data is an array of bytes.  Each byte represents a friction value, with 0 being the
 * natural friction of the touchscreen and 255 the highest friction achievable by the hardware.
 * The width times the height must equal the data size.
 *
 * @param ctx The context to use.
 * @param texture_id The texture to modify.
 * @param p_data The data to set.
 * @param width Texture width; must be non-negative.
 * @param height Texture height; must be non-negative.
 * @return In addition to the resource return values (see \ref resource-status), this function may
 * also return the following errors:
 *  - #TANVASTOUCH_ERROR_TEXTURE_NEGATIVE_WIDTH: The width is negative.
 *  - #TANVASTOUCH_ERROR_TEXTURE_NEGATIVE_HEIGHT: The height is negative.
 */
TANVASTOUCH_API int tanvastouch_set_texture_data(tanvastouch_ctx* ctx,
                                                 tanvastouch_resource_id texture_id,
                                                 const unsigned char* p_data, int width,
                                                 int height);

/**
 * Get a texture's dimensions.
 *
 * Texture dimensions are reported in pixels.
 *
 * @see \ref out-param-behavior
 * @param ctx The context to use.
 * @param texture_id The texture to query.
 * @param p_width Width storage.
 * @param p_height Height storage.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_get_texture_size(tanvastouch_ctx* ctx,
                                                 tanvastouch_resource_id texture_id, int* p_width,
                                                 int* p_height);

/**
 * Replace part of a texture's data with new data.
 *
 * @param ctx The context to use.
 * @param texture_id The texture to modify.
 * @param p_data The new data.
 * @param length The size of the new data.
 * @param offset The position in the existing texture data where replacement should start.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_set_texture_data_partial(tanvastouch_ctx* ctx,
                                                         tanvastouch_resource_id texture_id,
                                                         const unsigned char* p_data,
                                                         uint32_t length, uint32_t offset);

/**
 * Get the size of the data inside a texture.
 *
 * The size is returned in bytes.
 *
 * @see \ref out-param-behavior
 * @param ctx The context to use.
 * @param texture_id The texture to modify.
 * @param p_size Size storage.
 * @return \ref resource-status
 */
TANVASTOUCH_API int tanvastouch_get_texture_data_size(tanvastouch_ctx* ctx,
                                                      tanvastouch_resource_id texture_id,
                                                      uint32_t* p_size);

/** @} */

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* TANVAS_TANVASTOUCH_H */
